# -*- coding: utf-8 -*-
import bpy
import re
from bpy.props import BoolProperty, EnumProperty

# Addon情報
bl_info = {
    "name": "Switch Language",
    "author": "YdlProg",
    "version": (1, 2, 0),
    "blender": (4, 2, 0),
    "location": "",
    "description": "Key to switch language.(キーで言語の切り替え)",
    "warning": "",
    "doc_url": "https://ydlprog.ddns.net/2024/11/03/switch-language-%e3%82%ad%e3%83%bc%e3%81%a7%e8%a8%80%e8%aa%9e%e3%81%ae%e5%88%87%e3%82%8a%e6%9b%bf%e3%81%88blender-addon/",
    "category": "System",
}

# 翻訳辞書
translation_dict = {
    "ja_JP": {
        ("*", "1st Language"): "言語１",
        ("*", "2nd Language"): "言語２",
        (
            "*",
            "Important: `Keymap > Window > Switch Language` to change keymap.",
        ): "重要: キー割当は「キーマップ > ウィンドウ > Switch Language」で変更",
    }
}


# 対応言語をリストアップ
def get_languages():
    if bpy.app.version >= (4, 5, 0) and bpy.app.version <= (4, 5, 1):
        languages = []
        for entry in bpy.app.translations.locales:
            label = entry
            match = re.search(r"Locale code:\s*([\w@]+)", label)
            if match:
                locale_code = match.group(1)
                languages.append((locale_code, locale_code, ""))
        return languages
    else:
        return [(locale, locale, "") for locale in bpy.app.translations.locales]


languages = get_languages()


# Addon設定パネル
class SwitchLanguage_Preferences(bpy.types.AddonPreferences):
    bl_idname = __name__
    view = bpy.context.preferences.view

    # 言語1
    lang_1: EnumProperty(
        items=languages,
        name="1st Language",
        default="en_US",
    )
    # 言語2
    lang_2: EnumProperty(
        name="2nd Language",
        items=languages,
        default="ja_JP",
    )
    # ツールチップ
    trans_tooltips: BoolProperty(default=view.use_translate_tooltips, name="Tooltips")
    # インターフェイス
    trans_interface: BoolProperty(
        default=view.use_translate_interface, name="Interface"
    )
    # 報告
    trans_reports: BoolProperty(default=view.use_translate_reports, name="Reports")
    # 新規データ
    trans_new_dataname: BoolProperty(
        default=view.use_translate_new_dataname, name="New Data"
    )

    # パネル描画
    def draw(self, context):
        layout = self.layout
        layout.prop(self, "lang_1")
        layout.prop(self, "lang_2")
        row = layout.row()
        row.alignment = "EXPAND"
        row.prop(self, "trans_tooltips")
        row.prop(self, "trans_interface")
        row.prop(self, "trans_reports")
        row.prop(self, "trans_new_dataname")
        layout.label(
            text="Important: `Keymap > Window > Switch Language` to change keymap."
        )


class OBJECT_OT_SwitchLanguage(bpy.types.Operator):
    bl_label = "Switch Language"
    bl_idname = "object.switch_language"

    # 言語の切り替え
    def execute(self, context):
        prefs = bpy.context.preferences.addons[__name__].preferences
        view = bpy.context.preferences.view
        lang = view.language

        # 言語切替
        view.language = prefs.lang_2 if lang == prefs.lang_1 else prefs.lang_1
        # 翻訳範囲設定
        view.use_translate_tooltips = prefs.trans_tooltips
        view.use_translate_interface = prefs.trans_interface
        view.use_translate_reports = prefs.trans_reports
        view.use_translate_new_dataname = prefs.trans_new_dataname
        return {"FINISHED"}


# Blenderに登録するクラス
classes = [
    OBJECT_OT_SwitchLanguage,
    SwitchLanguage_Preferences,
]


# Addon有効化
def register():
    # クラス登録
    for c in classes:
        bpy.utils.register_class(c)
    # ショートカットキー登録
    kc = bpy.context.window_manager.keyconfigs.addon
    if kc:
        km = kc.keymaps.new(name="Window", space_type="EMPTY")
        kmi = km.keymap_items.new("object.switch_language", "END", "PRESS")
    # 翻訳辞書の登録
    bpy.app.translations.register(__name__, translation_dict)


# Addon無効化
def unregister():
    # クラス削除
    for c in classes:
        bpy.utils.unregister_class(c)
    # ショートカットキー削除
    kc = bpy.context.window_manager.keyconfigs.addon
    if kc:
        km = kc.keymaps["Window"]
        for kmi in km.keymap_items:
            if kmi.idname == "object.switch_language":
                km.keymap_items.remove(kmi)
                break
    # 翻訳辞書の登録解除
    bpy.app.translations.unregister(__name__)


# メイン処理
if __name__ == "__main__":
    register()
