# -*- coding: utf-8 -*-

import bpy
from bpy.props import BoolProperty, IntProperty, FloatProperty
from bpy.app.handlers import persistent

# Addon情報
bl_info = {
    "name": "Infinite time",
    "author": "YdlProg",
    "version": (1, 2, 0),
    "blender": (4, 2, 0),
    "location": "Timeline > Menu",
    "description": "Infinite time for drivers",
    "warning": "",
    "doc_url": "https://ydlprog.ddns.net/2025/01/28/%e6%b0%b8%e4%b9%85%e3%83%ab%e3%83%bc%e3%83%97%e3%81%99%e3%82%8btimer-node/",
    "category": "Animation",
}


# loop_countが更新された
def update_loop_count(self, context):
    if not bpy.context.scene.InfiniteTimeProperty.lock_loop_count:
        context.scene.frame_set(context.scene.frame_current)


# loop_countをリセット
class LoopResetOperator(bpy.types.Operator):
    bl_idname = "wm.loop_reset"
    bl_label = "Reset loop count"
    bl_options = {"REGISTER", "UNDO"}

    def execute(self, context):
        bpy.context.scene.InfiniteTimeProperty.loop_count = 0
        bpy.ops.screen.frame_jump(bpy.context.scene.frame_current)
        return {"FINISHED"}


# パネル情報プロパティ
class InfiniteTime_Property(bpy.types.PropertyGroup):
    # ループ数
    loop_count: IntProperty(default=0, min=-10000, max=10000, update=update_loop_count)
    # ループ数更新用のロック
    lock_loop_count: BoolProperty(default=False)
    # 最後に描画したフレーム
    last_play_frame: IntProperty(default=1, min=1)
    # UEのTimeノードと同じように進み続ける経過時間
    infinite_time: FloatProperty(default=0.0)


# タイムラインメニューにループ数とFPSを追加
def timeline_ui(self, context):
    scene = context.scene
    layout = self.layout
    row = layout.row(align=True)
    row.scale_x = 0.8
    row.prop(scene.InfiniteTimeProperty, "loop_count", text="Loop")
    layout.operator(LoopResetOperator.bl_idname, text="", icon="LOOP_BACK")
    layout.label(
        text=(
            " {:d} FPS".format(scene.render.fps)
            if scene.render.fps_base == 1.0
            else " {:.2f} FPS".format(scene.render.fps / scene.render.fps_base)
        )
    )


# 再生フレームが変化した
@persistent
def on_frame_change(scene):
    screen = (
        bpy.context.window_manager.windows[0].screen if bpy.context.window_manager.windows else None
    )
    if screen and screen.is_animation_playing and not screen.is_scrubbing:
        itp = scene.InfiniteTimeProperty
        frame = scene.frame_current
        # Lock
        itp.lock_loop_count = True
        # 再生
        if (frame - itp.last_play_frame) < -5:
            itp.loop_count += 1
            frame -= 1
        # 逆再生
        elif (frame - itp.last_play_frame) > 5:
            itp.loop_count -= 1
            frame += 1
        # Unlock
        itp.lock_loop_count = False
        itp.last_play_frame = frame

        infinite_time = itp.loop_count * scene.frame_end + scene.frame_current - 1
        itp.infinite_time = infinite_time / scene.render.fps


# Blenderに登録するクラス
classes = [InfiniteTime_Property, LoopResetOperator]


@persistent
def on_load_post(scene):
    # ドライバーの再登録
    bpy.app.driver_namespace["infinite_time"] = (
        lambda: bpy.context.scene.InfiniteTimeProperty.infinite_time
    )


# Addon有効化
def register():
    # クラス登録
    for c in classes:
        bpy.utils.register_class(c)
    # プロパティグループの登録
    bpy.types.Scene.InfiniteTimeProperty = bpy.props.PointerProperty(type=InfiniteTime_Property)
    if bpy.app.version >= (5, 0, 0):
        bpy.types.DOPESHEET_MT_editor_menus.append(timeline_ui)
    else:
        bpy.types.TIME_MT_editor_menus.append(timeline_ui)
    # ハンドラーの登録(フレームが変わった)
    bpy.app.handlers.frame_change_pre.append(on_frame_change)
    # ハンドラーの登録(ファイルが読み込まれた)
    bpy.app.handlers.load_post.append(on_load_post)
    # ドライバーの登録
    on_load_post(None)


# Addon無効化
def unregister():
    # クラス削除
    for c in classes:
        bpy.utils.unregister_class(c)
    # プロパティグループの削除
    delattr(bpy.types.Scene, "InfiniteTimeProperty")
    if bpy.app.version >= (5, 0, 0):
        bpy.types.DOPESHEET_MT_editor_menus.remove(timeline_ui)
    else:
        bpy.types.TIME_MT_editor_menus.remove(timeline_ui)
    # ハンドラーの削除
    bpy.app.handlers.frame_change_pre.remove(on_frame_change)
    bpy.app.handlers.load_post.remove(on_load_post)
    # ドライバーから削除
    del bpy.app.driver_namespace["infinite_time"]


# メイン処理
if __name__ == "__main__":
    register()
